/** @file   shadowmanager.cpp
 * @brief   Implementation of ShadowManager - class.
 * @version $Revision: 1.4 $
 * @author  Tomi Lamminsaari
 */

#include "shadowmanager.h"
#include "www_map.h"
#include "GfxManager.h"
#include "gfxid.h"
#include "shadowtable.h"
#include "settings.h"
using namespace eng2d;
using std::map;

namespace WeWantWar {


/** Constructor
 */
ShadowManager::ShadowManager()
{
}



/** Destructor
 */
ShadowManager::~ShadowManager()
{
  this->cleanup();
}


/** Builds the Dynamic Shadows
 */
void ShadowManager::buildShadows()
{
  for ( int y=0; y < Map::getHeight( Map::IN_PIXELS ); y += 256 ) {
    for ( int x=0; x < Map::getWidth( Map::IN_PIXELS ); x += 320 ) {
      int secnum = this->getSectorNumber( Vec2D(x,y) );
      Rect2D rect( Vec2D(x,y), Vec2D(x+289, y+225) );
      iShadows[ secnum ] = new ShadowTable( rect );
    }
  }
}



/** Applies the shadows to the given background buffer.
 */
void ShadowManager::applyShadows( BITMAP* aTarget, const Vec2D& aCenter )
{
  if ( Settings::wallShadows == false ) {
    return;
  }
  
  Vec2D sectors[ 9 ];
  sectors[0].set( aCenter.vx - 320, aCenter.vy - 256 );
  sectors[1].set( aCenter.vx, aCenter.vy - 256 );
  sectors[2].set( aCenter.vx + 320, aCenter.vy - 256 );
  
  sectors[3].set( aCenter.vx - 320, aCenter.vy );
  sectors[4].set( aCenter.vx, aCenter.vy );
  sectors[5].set( aCenter.vx + 320, aCenter.vy );
  
  sectors[6].set( aCenter.vx - 320, aCenter.vy + 256 );
  sectors[7].set( aCenter.vx, aCenter.vy + 256 );
  sectors[8].set( aCenter.vx + 320, aCenter.vy + 256 );

  // We apply the shadows to the sectors around us
  for ( int i=0; i < 9; i++ ) {
    int secnum = this->getSectorNumber( sectors[i] );
    map< int, ShadowTable* >::const_iterator it = iShadows.find( secnum );
    if ( it != iShadows.end() ) {
      it->second->applyShadows( aTarget );
    }
  }
}



/** Removes all the ShadowTable - objects there are.
 */
void ShadowManager::cleanup()
{
  map< int, ShadowTable* >::const_iterator it = iShadows.begin();
  while ( it != iShadows.end() ) {
    delete it->second;
    it++;
  }
  iShadows.clear();
}


/** Returns the sector number
 */
int ShadowManager::getSectorNumber( const Vec2D& aP ) const
{
  int px = aP.intX();
  int py = aP.intY();

  if ( px < 0 || py < 0 ) {
    return -1;
  }
  
  px /= 320;
  py /= 256;
  return py * 100 + px;
}


} // end of namespace
